<?php

namespace Rawbinn\Media\Http\Controllers;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;
use Intervention\Image\Facades\Image;
use Rawbinn\Media\Facades\Media;
use Rawbinn\Media\Models\Media as MediaModel;

/**
 * Class MediaController.
 *
 * @version 1.0.0
 *
 * @author Rawbinn Shrestha <contact@rawbinn.com>
 * @organization RAWBINN.COM.
 */
class MediaController extends Controller
{
    /**
     * @var \Illuminate\Config\Repository|mixed
     */
    protected $disk;

    /**
     * MediaController constructor.
     */
    public function __construct()
    {
        $this->disk = config('filesystems.default');
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function upload(Request $request)
    {
        $type = config('media.supported');
        $max_file_size = config('media.max_file_size');
        $resize = config('media.resize') ?? false;
        $rename = config('media.rename') ?? false;
        $thumbnail = config('media.thumbnail');
        $dimensions = config('media.dimensions.default') ?? ['100' => '100'];

        if ($request->hasFile('file')) {
            $data = [];
            $data['file_original_name'] = null;

            /* Extension validation */
            $data['extension'] = strtolower($request->file('file')->getClientOriginalExtension());
            if (!array_key_exists($data['extension'], $type)) {
                return response()->json(['status' => false, 'message' => 'File not supported'], 415);
            }

            $filetype = explode('/', $request->type);
            $filetype = $filetype[0] == 'application' ? 'document' : $filetype[0];
            $specificType = array_filter($type, function($type) use ($filetype) {
                return $type == $filetype;
            });
           
            if (!array_key_exists($data['extension'], $specificType)) {
                return response()->json(['status' => false, 'message' => 'File not supported'], 415);
            }

            /* File size validation */
            $data['file_size'] = $request->file('file')->getSize();
            if ($data['file_size'] > $max_file_size) {
                return response()->json(['status' => false, 'message' => 'File size too large'], 415);
            }

            $arr = explode('.', $request->file('file')->getClientOriginalName());

            for ($i = 0; $i < count($arr) - 1; ++$i) {
                if ($i == 0) {
                    $data['file_original_name'] .= $arr[$i];
                } else {
                    $data['file_original_name'] .= '.'.$arr[$i];
                }
            }
            $folder = ($request->has('group') && $request->group != '') ? $request->group : config('media.folder');

            /** Destination upload folder */
            $destination_folder = md5(Auth::id()).DIRECTORY_SEPARATOR.$folder.DIRECTORY_SEPARATOR.date('Y-m');

            if ($rename) {
                $data['path'] = $request->file('file')->store($destination_folder, $this->disk);
            } else {
                $filename = substr(Str::slug($data['file_original_name']), 0, 20).'-'.time().'.'.$data['extension'];
                $data['path'] = $request->file('file')->storeAs($destination_folder, $filename, $this->disk);
            }

            $data['file_name'] = last(explode('/', $data['path']));
            $data['user_id'] = Auth::id();

            if (isset($type[$data['extension']])) {
                $data['type'] = $type[$data['extension']];
            } else {
                $data['type'] = 'others';
            }

            $data['groups'] = $request->group != '' ? $request->group : config('media.folder');

            $media = MediaModel::create($data);

            if ($thumbnail != null) {
                $image = Image::make(public_path('storage/'.$data['path']));
                $image->resize(array_key_first($thumbnail), array_values($thumbnail)[0], function ($constraint) {
                    $constraint->aspectRatio();
                    $constraint->upsize();
                })->encode($data['extension']);
                $filename = explode('.', $data['file_name']);
                $thumbnail_file_name = $filename[0].'-thumbnail.'.$data['extension'];
                Storage::disk($this->disk)->put($destination_folder.DIRECTORY_SEPARATOR.$thumbnail_file_name, $image);
            }

            if ($resize) {
                /* Custom dimension according to group */
                if (in_array($request->group, array_keys(config('media.dimensions')))) {
                    $dimensions = config('media.dimensions.'.$request->group);
                }
                $image = Image::make(public_path('storage/'.$data['path']));
                $image->backup();
                foreach ($dimensions as $width => $height) {
                    $filename = explode('.', $data['file_name']);
                    $resize_file_name = $filename[0].'-'.$width.'x'.$height.'.'.$data['extension'];
                    $width = $width == 'auto' ? null : $width;
                    $height = $height == 'auto' ? null : $height;
                    $image->resize($width, $height, function ($constraint) {
                        $constraint->aspectRatio();
                        $constraint->upsize();
                    })->encode($data['extension']);

                    Storage::disk($this->disk)->put($destination_folder.DIRECTORY_SEPARATOR.$resize_file_name, $image);
                    $image->reset();
                }
            }

            return response()->json(['id' => $media->id, 'url' => $media->url]);
        }
    }

    /**
     * @return mixed
     *
     * @author Rawbinn Shrestha ( rawbinnn@gmail.com )
     */
    public function files(Request $request)
    {
        $user_id = Auth::id();

        $media = MediaModel::where('user_id', $user_id);
        if ($request->search != null) {
            $media->where('file_original_name', 'like', '%'.$request->search.'%');
        }
        if ($request->sort != null) {
            switch ($request->sort) {
                case 'newest':
                    $media->orderBy('created_at', 'desc');
                    break;
                case 'oldest':
                    $media->orderBy('created_at', 'asc');
                    break;
                case 'smallest':
                    $media->orderBy('file_size', 'asc');
                    break;
                case 'largest':
                    $media->orderBy('file_size', 'desc');
                    break;
                default:
                    $media->orderBy('created_at', 'desc');
                    break;
            }
        }
        if ($request->has('group') && $request->group != '') {
            $media = $media->where('groups', $request->group);
        }
        if($request->has('type')) {
            $media = $media->where('type', $request->type);
        }

        return $media->paginate(60)->appends(request()->query());
    }

    /**
     * Display the specified resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function show()
    {
        return view('media::filemanager');
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function preview(Request $request)
    {
        $ids = explode(',', $request->ids);
        $files = $this->getByIds($ids);

        return response()->json($files);
    }

    /**
     * Download project attachment.
     *
     * @param int $id
     *
     * @return \Illuminate\Http\Response
     */
    public function download($id)
    {
        $project_attachment = MediaModel::find($id);
        try {
            $file_path = public_path($project_attachment->path);

            return response()->download($file_path);
        } catch (\Exception $e) {
            flash(translate('File does not exist!'))->error();

            return back();
        }
    }

    public function imageUpload(Request $request)
    {
        $type = config('media.supported');
        $max_file_size = config('media.max_file_size');
        $resize = config('media.resize') ?? false;
        $thumbnail = config('media.thumbnail');
        $dimensions = config('media.dimensions.default') ?? ['100' => '100'];
        $shared = config('media.shared');
        if ($request->hasFile('image')) {
            $data = [];
            $data['file_original_name'] = null;

            $data['extension'] = strtolower($request->file('image')->getClientOriginalExtension());
            if (!array_key_exists($data['extension'], $type)) {
                return response()->json(['status' => false, 'message' => 'File not supported'], 415);
            }
            $data['file_size'] = $request->file('image')->getSize();
            if ($data['file_size'] > $max_file_size) {
                return response()->json(['status' => false, 'message' => 'File size too large'], 415);
            }
            $filename = $request->file('image')->getClientOriginalName();

            $filename = explode($data['extension'], strtolower($filename));
            $data['file_name'] = Str::slug($filename[0]).'-original-'.date('his').'.'.$data['extension'];

            $folder = ($request->has('group') && $request->group != '') ? $request->group : config('media.folder');

            $data['groups'] = $request->group != '' ? $request->group : config('media.folder');

            /* Destination upload folder */
            if (in_array($data['groups'], $shared)) {
                $destination_folder = 'shared'.DIRECTORY_SEPARATOR.$folder.DIRECTORY_SEPARATOR.date('Y-m');
            } else {
                $destination_folder = md5(Auth::id()).DIRECTORY_SEPARATOR.$folder.DIRECTORY_SEPARATOR.date('Y-m');
            }

            // Storage::disk($this->disk)->put($destination_folder.DIRECTORY_SEPARATOR.$filename, $request->file('image'));
            $data['path'] = '';
            if ($request->hasFile('image')) {
                $data['path'] = $request->image->storeAs($destination_folder, $data['file_name'], $this->disk);
            }
            $filename = explode('.', $data['file_name']);
            $data['variants'][] = public_asset('storage/'.str_replace('\\', '/', $destination_folder.DIRECTORY_SEPARATOR.$filename[0].'.'.$data['extension']));
            if ($resize) {
                /* Custom dimension according to group */
                if (in_array($request->group, array_keys(config('media.dimensions')))) {
                    $dimensions = config('media.dimensions.'.$request->group);
                }
                $image = Image::make(public_path('storage/'.$data['path']));
                $image->backup();
                foreach ($dimensions as $width => $height) {
                    $resize_file_name = $filename[0].'-'.$width.'x'.$height.'.'.$data['extension'];
                    $width = $width == 'auto' ? null : $width;
                    $height = $height == 'auto' ? null : $height;
                    $image->resize($width, $height, function ($constraint) {
                        $constraint->aspectRatio();
                        $constraint->upsize();
                    })->encode($data['extension']);

                    Storage::disk($this->disk)->put($destination_folder.DIRECTORY_SEPARATOR.$resize_file_name, $image);
                    $image->reset();
                    $data['variants'][] = public_asset('storage/'.str_replace('\\', '/', $destination_folder.DIRECTORY_SEPARATOR.$resize_file_name));
                }
            }

            return response()->json(['original_url' => public_asset('storage/'.str_replace('\\', '/', $data['path'])), 'thumbnail_url' => $data['variants'][0]]);
        }
    }

    public function getCacheImage($path, $dimensions = null)
    {
        return Media::cache($path, $dimensions);
    }

    public function getByIds($ids)
    {
        return MediaModel::select('id', 'file_original_name', 'file_size', 'extension', 'type', 'path')
        ->orderBy('id', 'DESC')
        ->whereIn('id', $ids)->get();
    }
}
