<?php namespace App\Services;

use Illuminate\Support\Collection;
use Illuminate\Database\Eloquent\Builder;


/**
 * trait Slugify
 * @package App\Services
 * @author Rawbinn Shrestha <rawbinnn@gmail.com>
 */
trait Sluggable
{
    public static function bootSluggable()
	{
		static::saving(function ($model) {
            $settings = $model->sluggable();
            $source = $settings['source'];
            if($model->slug != null) {
                $source = 'slug';
            }
			$model->slug = $model->slugify($model->$source);
		});
	}

	abstract public function sluggable(): array;

    public function slugify($string)
    {
        $config = config('sluggable');
        if ($this->getKey() > 0) { 
            if($this->slug != $this->getOriginal('slug') ){
                $slug = $this->generateSlug($this->slug, $config);
                $slug = $this->makeSlugUnique($slug, $config);
                return $slug;
            }
            return $this->slug;
        }
       

        $slug = $this->generateSlug($string, $config);
        $slug = $this->makeSlugUnique($slug, $config);
        return $slug;
    }

	public function generateSlug($slug, array $config)
	{
        $config = config('sluggable');
        $separator = $config['separator'];
        $method = $config['method'];
        $maxLength = $config['maxLength'];
        $maxLengthKeepWords = $config['maxLengthKeepWords'];

        $slug = str_replace(array('\'', '"'), '', mb_strtolower($slug));
        $slug = preg_replace('/\s+/', $separator, $slug);

        $len = mb_strlen($slug);
        if (is_string($slug) && $maxLength && $len > $maxLength) {
            $reverseOffset = $maxLength - $len;
            $lastSeparatorPos = mb_strrpos($slug, $separator, $reverseOffset);
            if ($maxLengthKeepWords && $lastSeparatorPos !== false) {
                $slug = mb_substr($slug, 0, $lastSeparatorPos);
            } else {
                $slug = trim(mb_substr($slug, 0, $maxLength), $separator);
            }
        }
        return $slug;
    }

    /**
     * Checks if the slug should be unique, and makes it so if needed.
     *
     * @param string $slug
     * @param array $config
     * @param string $attribute
     *
     * @return string
     * @throws \UnexpectedValueException
     */
    protected function makeSlugUnique(string $slug, array $config): string
    {
        if (!$config['unique']) {
            return $slug;
        }
        $separator = $config['separator'];
        // find all models where the slug is like the current one
        $list = $this->getExistingSlugs($slug, $config);
        // if ...
        // 	a) the list is empty, or
        // 	b) our slug isn't in the list
        // ... we are okay
        if (
            $list->count() === 0 ||
            $list->contains($slug) === false
        ) {
            return $slug;
        };
        // if our slug is in the list, but
        // 	a) it's for our model, or
        //  b) it looks like a suffixed version of our slug
        // ... we are also okay (use the current slug)
        // if ($list->has($this->getKey())) {
        //     $currentSlug = $list->get($this->getKey());
        //     if (
        //         $currentSlug === $slug ||
        //         strpos($currentSlug, $slug) === 0
        //     ) {
        //         return $currentSlug;
        //     }
        // }
        
        $suffix = $this->generateSuffix($slug, $separator, $list);
        return $slug . $separator . $suffix;
    }
    
    /**
     * Get all existing slugs that are similar to the given slug.
     *
     * @param string $slug
     * @param array $config
     *
     * @return \Illuminate\Support\Collection
     */
    protected function getExistingSlugs(string $slug, array $config): Collection
    {
        $includeTrashed = $config['includeTrashed'];
        $query = $this->findSimilarSlugs($config, $slug);
        // include trashed models if required
        // dd($this);
        $traits = class_uses($this);
        if (in_array('Illuminate\Database\Eloquent\SoftDeletes', $traits) & $includeTrashed)
        {
            $query->withTrashed();
        }
        // get the list of all matching slugs
        $results = $query->select(['slug'])
            ->get()
            ->toBase();
        // key the results and return
        return $results->pluck('slug');
    }

    public function findSimilarSlugs(array $config, string $slug): Builder
    {
        $separator = $config['separator'];
        return $this->where(function(Builder $q) use ($slug, $separator) {
            $q->where('slug', '=', $slug)
                ->orWhere('slug', 'LIKE', $slug . $separator . '%');
        });

    }

    /**
     * Generate a unique suffix for the given slug (and list of existing, "similar" slugs.
     *
     * @param string $slug
     * @param string $separator
     * @param \Illuminate\Support\Collection $list
     *
     * @return string
     */
    protected function generateSuffix(string $slug, string $separator, Collection $list): string
    {
        $len = mb_strlen($slug . $separator);;
        // If the slug already exists, but belongs to
        // our model, return the current suffix.
        // dd($this->getKey());
        if ($list->search($slug) === $this->getKey()) {
            $suffix = explode($separator, $slug);
            // dd('insite');
            return end($suffix);
        }
        // dd('ding');
        $list->transform(function($value, $key) use ($len) {
            return (int) mb_substr($value, $len);
        });
        // find the highest value and return one greater.
        return $list->max() + 1;
    }

}