<?php
namespace App\Repositories\Backend;

use Module;
use Artisan;
use App\Services\Zipper;
use App\Exceptions\GeneralException;

/**
 * Class PackageRepository
 * @package App\Repositories\Backend
 * @author Rawbinn Shrestha <rawbinnn@gmail.com>
 */
class PackageRepository
{
    /**
     * Get packages name from package folder
     *
     * @return void
     */
    public function all()
    {
        try {
            $lists = scandir(config('modules.locations.app.path'));
        }catch(\Exception $e){
            echo $e->getMessage();
            echo '<a href="'.route('admin.dashboard').'"> Redirect to home</a>';
            exit();
        }
        unset($lists[0]);
        unset($lists[1]);
        $key = array_search('index.html', $lists);
        unset($lists[$key]);
        return array_values($lists);
    }

    /**
     * Get all packages and refresh package cache
     *
     * @return void
     */
    public function getModules()
    {
        $live_modules = $this->all();
        $cached_modules = Module::all()->keys()->toArray();
        if(count($live_modules) > count($cached_modules)) {
            if(count(array_diff($live_modules, $cached_modules)) > 0) {
                Module::optimize();
            }
        }
        else {
            if (count(array_diff($cached_modules, $live_modules)) > 0) {
                Module::optimize();
            }
        }
        return Module::all();
    }

    /**
     * Enable package
     *
     * @param string $package_name
     * @return void
     */
    public function enable($package_name)
    {
        if (Module::exists($package_name)) {
            Module::enable($package_name);
        }
    }

    /**
     * Disable package
     *
     * @param string $package_name
     * @return void
     */
    public function disable($package_name)
    {
        if (Module::exists($package_name)) {
            Module::disable($package_name);
        }
    }

    /**
     * Delete package
     *
     * @param string $package_name
     * @return void
     */
    public function delete($package_name)
    {
        $package_path = config('modules.path').'/'.ucfirst($package_name);
        if (Module::exists($package_name)) {
            Artisan::call('module:migrate:rollback', ['slug' => $package_name, '--force' => true]);
            $this->deleteAllFiles($package_path);
            Module::optimize();
        }
        return true;
    }

    /**
     * Extract package
     *
     * @param string $path
     * @return boolean
     */
    public function extract($path)
    {
        $file_path = storage_path('app/' . $path);
        $package_path = config('modules.path');
        $zip = Zipper::open($file_path);
        $dir_name_inside_zip = ucfirst(rtrim($zip->getNameIndex(0), '/'));

        $full_package_path = $package_path . '/' . $dir_name_inside_zip;
        if (file_exists($full_package_path)) {
            throw new GeneralException('Package already exist with this name.');
        }
        # Check if theme has module.json,assets, views file or not
        if ( $zip->locateName($dir_name_inside_zip . '/module.json') ) {
            $zip->extract($package_path);
            $zip->close();
            unlink($file_path);
            try{
                $this->loadPackage($dir_name_inside_zip);
            }
            catch(\Exception $e)
            {
                $this->deleteAllFiles($full_package_path);
                throw new GeneralException($e->getMessage());
                return false;
            }
            return true;
        } else {
            unlink($file_path);
            throw new GeneralException('Invalid package, module.json file not found in package.');
            return false;
        }

    }

    /**
     * Load package , it runs artisan of its package
     *
     * @param string $package_name
     * @return boolean
     */
    public function loadPackage($package_name)
    {
        $package_name = strtolower($package_name);
        
        $manifest = Module::getManifest($package_name);
        if(!array_key_exists('slug', $manifest->toArray())) throw new \Exception('Invalid package, module.json file is not valid.');
        Module::optimize();
        Artisan::call('module:migrate', ['slug' => $package_name, '--force' => true]);
        return true;
    }

    /**
     * Recursive delete all directory and its files of given directory
     *
     * @param string $dir
     * @return void
     */
    public function deleteAllFiles($dir)
    {
        if (!$dh = @opendir($dir)) {
            return;
        }

        while (false !== ($obj = readdir($dh))) {
            if ($obj == '.' || $obj == '..') {
                continue;
            }

            if (!@unlink($dir . '/' . $obj)) {
                $this->deleteAllFiles($dir . '/' . $obj, true);
            }

        }
        closedir($dh);
        @rmdir($dir);

    }

}
