<?php

namespace App\Repositories\Backend;

use App\Events\Backend\Blog\BlogCreated;
use App\Events\Backend\Blog\BlogDeleted;
use App\Events\Backend\Blog\BlogUpdated;
use App\Exceptions\GeneralException;
use App\Models\Post\Post;
use App\Repositories\Repository;
use DateTime;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\DB;

/**
 * Class BlogRepository.
 *
 * @author Rawbinn Shrestha <rawbinnn@gmail.com>
 */
class BlogRepository extends Repository
{
    /**
     * Associated Repository Model.
     */
    const MODEL = Post::class;

    /**
     * @param bool $trashed
     *
     * @return mixed
     */
    public function getForDataTable($trashed = false)
    {
        /**
         * Note: You must return deleted_at or the User getActionButtonsAttribute won't
         * be able to differentiate what buttons to show for each row.
         */
        $dataTableQuery = $this->query()
            ->select([
                'id',
                'slug',
                'user_id',
                'type',
                'status',
                'featured',
                'meta_title',
                'meta_description',
                'created_at',
                'updated_at',
                'deleted_at',
            ])
            ->where('type', 'post')
            ->with(['user', 'categories']);

        if (!auth()->user()->hasRole(1)) {
            $dataTableQuery->where('user_id', auth()->id());
        }

        $dataTableQuery = filter('larapress.blog.datatables.query', $dataTableQuery);

        if ($trashed == 'true') {
            return $dataTableQuery->onlyTrashed();
        }
        return $dataTableQuery;
    }

    /**
     * @return mixed
     */
    public function getAll($orderBy = 'ASC')
    {
        return $this->query()->where('type', 'post')->where('status', '1')->orderBy('id', $orderBy)->get();
    }

    /**
     * @throws GeneralException
     */
    public function create(array $attributes)
    {
        $data = [
            'type' => 'post',
            'media_id' => ($attributes['media_id'] != '') ? $attributes['media_id'] : null,
            'user_id' => access()->id(),
            'status' => $attributes['status'],
            'featured' => (array_key_exists('featured', $attributes) ? true : false),
            'meta_title' => ($attributes['meta_title'] != '') ? $this->filterString($attributes['meta_title']) : $this->filterString($attributes['title'] ?? ''),
            'meta_description' => ($attributes['meta_description'] != '') ? $this->filterString($attributes['meta_description'], 160) : $this->filterString($attributes['content'] ?? '', 160),
            'created_at' => (isset($attributes['date']) && validateDate($attributes['date'])) ? $attributes['date'] : date('Y-m-d H:i:s'),
            'updated_at' => (isset($attributes['date']) && validateDate($attributes['date'])) ? $attributes['date'] : date('Y-m-d H:i:s')
        ];
        if (new DateTime() < new DateTime($data['created_at'])) {
            $data['status'] = 0;
        }
        $model = self::MODEL;
        $model = new $model();

        if (!array_key_exists('category', $attributes)) {
            $attributes['category'] = 1;
        }

        try {
            foreach (locales() as $locale) {
                foreach ($model->translatedAttributes as $attribute) {
                    if (isset($attributes[$attribute])) {
                        $data[$locale][$attribute] = $attributes[$attribute];
                    }
                }
            }
            $model->fill($data);
            parent::save($model);
            $model->categories()->attach($attributes['category']);
            if ($attributes['tags'] != null) {
                $model->attachTags($attributes['tags']);
            }
            filter('larapress.blog.create.end', ['blog' => $model, 'inputs' => $attributes]);
        } catch (\Exception $e) {
            // throw new GeneralException($e->getMessage());
            throw new GeneralException(trans('exceptions.backend.blog.create_error'));
        }
        event(new BlogCreated($model, $attributes));
        return $model;
    }

    public function update(Model $blog, array $attributes)
    {
        if (!auth()->user()->hasRole(1) && auth()->user()->hasRole(3) && auth()->id() != $blog->user_id) {
            throw new GeneralException('You do not have permission to edit this post.');
        }

        $data = [
            'slug' => $attributes['slug'],
            'status' => $attributes['status'],
            'featured' => (array_key_exists('featured', $attributes) ? true : false),
            'meta_title' => ($attributes['meta_title'] != '') ? $this->filterString($attributes['meta_title']) : $this->filterString($attributes['en']['title'] ?? ''),
            'meta_description' => ($attributes['meta_description'] != '') ? $this->filterString($attributes['meta_description'], 160) : $this->filterString($attributes['en']['content'] ?? '', 160),
            'created_at' => (isset($attributes['date']) && validateDate($attributes['date'])) ? $attributes['date'] : date('Y-m-d H:i:s')
        ];
        if (new DateTime() < new DateTime($data['created_at'])) {
            $data['status'] = 0;
        }
        $data[$attributes['lang']] = $attributes[$attributes['lang']];

        if ($attributes['media_id'] != '0') {
            $data['media_id'] = ($attributes['media_id'] != '') ? $attributes['media_id'] : null;
        }
        if (!array_key_exists('category', $attributes)) {
            $data['category'] = 1;
        }
        try {
            if (parent::update($blog, $data)) {
                $blog->categories()->sync($attributes['category']);
                if ($attributes['tags'] != null) {
                    $blog->syncTags($attributes['tags']);
                }
                filter('larapress.blog.update.end', ['blog' => $blog, 'inputs' => $attributes]);
            }
        } catch (\Exception $e) {
            throw new GeneralException(trans('exceptions.backend.blog.update_error'));
        }
        event(new BlogUpdated($blog, $attributes));
        return true;
    }

    /**
     * @return bool
     *
     * @throws GeneralException
     */
    public function delete(Model $blog)
    {
        if (parent::delete($blog)) {
            event(new BlogDeleted($blog));
            return true;
        }

        throw new GeneralException(trans('exceptions.backend.blog.delete_error'));
    }

    /**
     * @throws GeneralException
     */
    public function forceDelete(Model $blog)
    {
        DB::transaction(function () use ($blog) {
            $blog->categories()->detach();
            if (parent::forceDelete($blog)) {
                $blog->deleteTranslations();
                $blog->metas()->delete();
                event(new BlogDeleted($blog));
                return true;
            }

            throw new GeneralException(trans('exceptions.backend.blog.delete_error'));
        });
    }

    /**
     * @return bool
     *
     * @throws GeneralException
     */
    public function restore(Model $blog)
    {

        if (parent::restore(($blog))) {
            if ($blog->categories->isEmpty()) {
                $blog->categories()->attach(1);
            }
            return true;
        }

        throw new GeneralException(trans('exceptions.backend.blog.restore_error'));
    }

    /**
     * Filter the string.
     *
     * @param string $string
     *
     * @return string
     */
    public function filterString($string, $length = 63)
    {
        $string = str_replace(PHP_EOL, '', $string);

        return mb_substr(strip_tags($string), 0, $length);
    }
}
