<?php

namespace App\Models\Post;

use DB;

/**
 * Class PostAttribute.
 *
 * @author Rawbinn Shrestha <rawbinnn@gmail.com>
 */
trait PostAttribute
{
    public function hasFeaturedImage()
    {
        if ($this->media_id) {
            return true;
        }

        return false;
    }

    public function getFeaturedImage($thumbnail = false)
    {
        $image = DB::table(config('media.table'))->where('id', $this->media_id)->first();
        if ($image) {
            return public_asset('storage/'.str_replace('\\', '/', $image->path));
        }

        return '';
    }

    public function featuredImage($thumbnail = false)
    {
        $image = DB::table(config('media.table'))->where('id', $this->media_id)->first();
        if ($image) {
            return public_asset('storage/' . str_replace('\\', '/', $image->path));
        }

        return '';
    }

    public function scopeActive($query)
    {
        return $query->where('status', '1');
    }

    public function scopeType($query, $type)
    {
        return $query->where('type', $type);
    }

    public function scopeTemplate($query, $template)
    {
        return $query->where('template', $template);
    }

    /**
     * @return string Return title of post
     */
    public function title()
    {
        return $this->title;
    }

    /**
     * @return string Return title of post
     */
    public function subtitle()
    {
        return $this->subtitle;
    }

    /**
     * @return string Return content of post
     */
    public function content($count = null, $readmore = false, $end = null)
    {
        if ($count != null) {
            return excerpt($this->content, $count, $readmore, $end);
        }

        return $this->content;
    }

    /**
     * @return string
     *                Return permalink of post
     */
    public function permalink()
    {
        return route('post', $this->slug);
    }

    /**
     * @return string Return date of post
     */
    public function date($format = 'M d, Y', $type = 'created')
    {
        if ($type == 'updated') {
            return $this->updated_date->format($format);
        }

        return $this->created_at->format($format);
    }

    /**
     * @return string Return created date of post
     */
    public function created_date($format = 'M d, Y')
    {
        return $this->created_at->format($format);
    }

    /**
     * @return string Return updated date of post
     */
    public function updated_date($format = 'M d, Y')
    {
        return $this->updated_at->format($format);
    }

    /**
     * @return string Return categories of post
     */
    public function getCategories()
    {
        return implode(', ', $this->categories->pluck('title')->toArray());
    }

    /**
     * @return string Return author of post
     */
    public function author($attribute = null)
    {
        if ($attribute == 'firstname') {
            return $this->user->profile->firstname;
        } elseif ($attribute == 'lastname') {
            return $this->user->profile->lastname;
        } elseif ($attribute == 'lastname:firstname') {
            return $this->user->profile->lastname.' '.$this->user->profile->firstname;
        } else {
            return $this->user->profile->firstname.' '.$this->user->profile->lastname;
        }
    }

    /**
     * @return string Return total comments count of post
     */
    public function totalComments()
    {
        return $this->comments->count();
    }

    public function previous()
    {
        return Post::where('id', '<', $this->id)->orderBy('id', 'ASC')->first();
    }

    public function next()
    {
        return Post::where('id', '>', $this->id)->orderBy('id', 'DESC')->first();
    }

    /**
     * @return string
     */
    public function getShowButtonAttribute()
    {
        return '<a href="'.route('post', $this->slug).'" target="_blank" class="btn btn-xs btn-info"><i class="fa fa-eye" data-toggle="tooltip" data-placement="top" title="'.trans('buttons.general.crud.view').'"></i></a> ';
    }

    /**
     * @return string
     */
    public function getEditButtonAttribute()
    {
        if ($this->type == 'page' && auth()->user()->hasPermission('edit-pages')) {
            return '<a href="'.route('admin.page.edit', $this).'" class="btn btn-xs btn-primary"><i class="fa fa-pencil-alt" data-toggle="tooltip" data-placement="top" title="'.trans('buttons.general.crud.edit').'"></i></a> ';
        }
        if ( $this->type == 'post' && auth()->user()->hasPermission('edit-blogs')) {
            return '<a href="'.route('admin.blog.edit', $this).'" class="btn btn-xs btn-primary"><i class="fa fa-pencil-alt" data-toggle="tooltip" data-placement="top" title="'.trans('buttons.general.crud.edit').'"></i></a> ';
        }
    }

    /**
     * @return string
     */
    public function getDeleteButtonAttribute()
    {
        if ($this->type == 'page' && auth()->user()->hasPermission('delete-pages')) {
            return '<a href="'.route('admin.page.destroy', $this).'"
                 data-method="delete"
                 data-trans-button-cancel="'.trans('buttons.general.cancel').'"
                 data-trans-button-confirm="'.trans('buttons.general.crud.delete').'"
                 data-trans-title="'.trans('strings.backend.general.are_you_sure').'"
                 class="btn btn-xs btn-danger"><i class="fa fa-trash" data-toggle="tooltip" data-placement="top" title="Trash"></i></a> ';
        }
        if ($this->type == 'post' && auth()->user()->hasPermission('delete-blogs')) {
            return '<a href="'.route('admin.blog.destroy', $this).'"
                    data-method="delete"
                    data-trans-button-cancel="'.trans('buttons.general.cancel').'"
                    data-trans-button-confirm="'.trans('buttons.general.crud.delete').'"
                    data-trans-title="'.trans('strings.backend.general.are_you_sure').'"
                    class="btn btn-xs btn-danger"><i class="fa fa-trash" data-toggle="tooltip" data-placement="top" title="Trash"></i></a> ';
        }
    }

    /**
     * @return string
     */
    public function getRestoreButtonAttribute()
    {
        if ($this->type == 'page') {
            return '<a href="'.route('admin.page.restore', $this).'" name="restore_page" class="btn btn-xs btn-info"><i class="fa fa-redo" data-toggle="tooltip" data-placement="top" title="'.trans('buttons.backend.blog.restore_blog').'"></i></a> ';
        }

        return '<a href="'.route('admin.blog.restore', $this).'" name="restore_blog" class="btn btn-xs btn-info"><i class="fa fa-redo" data-toggle="tooltip" data-placement="top" title="'.trans('buttons.backend.blog.restore_blog').'"></i></a> ';
    }

    /**
     * @return string
     */
    public function getDeletePermanentlyButtonAttribute()
    {
        if ($this->type == 'page') {
            return '<a href="'.route('admin.page.delete-permanently', $this).'" name="delete_page_perm" class="btn btn-xs btn-danger"><i class="fa fa-trash" data-toggle="tooltip" data-placement="top" title="'.trans('buttons.backend.general.delete_permanently').'"></i></a> ';
        }

        return '<a href="'.route('admin.blog.delete-permanently', $this).'" name="delete_blog_perm" class="btn btn-xs btn-danger"><i class="fa fa-trash" data-toggle="tooltip" data-placement="top" title="'.trans('buttons.backend.general.delete_permanently').'"></i></a> ';
    }

    /**
     * @return string
     */
    public function getActionButtonsAttribute()
    {
        if ($this->trashed()) {
            return $this->getRestoreButtonAttribute().
                $this->getDeletePermanentlyButtonAttribute();
        }

        return
            $this->getShowButtonAttribute().
            $this->getEditButtonAttribute().
            $this->getDeleteButtonAttribute();
    }
}
