<?php
/**
 * Class Post.
 *
 * @author Rawbinn Shrestha <rawbinnn@gmail.com>
 * @date 12/20/16
 * @time 8:08 PM
 */

namespace  App\Models\Post;

use App\Models\Category\Category;
use App\Models\Comment\Comment;
use App\Models\Tag\Tag;
use App\Services\Sluggable;
use App\Services\SluggableScopeHelpers;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Spatie\Feed\Feedable;
use Spatie\Feed\FeedItem;
use Astrotomic\Translatable\Contracts\Translatable as TranslatableContract;
use Astrotomic\Translatable\Translatable;
use Illuminate\Support\Arr;
use Spatie\Sitemap\Contracts\Sitemapable;
use Spatie\Sitemap\Tags\Url;


class Post extends Model implements Feedable, TranslatableContract, Sitemapable
{
    use PostAttribute;
    use PostMetaRelation;
    use Sluggable;
    use SoftDeletes;
    use SluggableScopeHelpers;
    use Translatable;
    
    public $translatedAttributes = ['title', 'subtitle', 'content'];
    
    /**
     * The database table used by the model.
     *
     * @var string
     */
    protected $table;

    /**
     * The attributes that are mass assignable.
     *
     * @var array
     */
    protected $guarded = ['id'];

    /**
     * @var array
     */
    protected $dates = ['deleted_at'];

    public function __construct(array $attributes = [])
    {
        parent::__construct($attributes);
        $this->table = 'posts';
    }

    public function sluggable() : array
    {
        return [
            'source' => 'title',
        ];
    }

    /**
     * Return the categories.
     */
    public function categories()
    {
        return $this->belongsToMany('App\Models\Category\Category');
    }

    public function category()
    {
        return $this->belongsTo(Category::class)->withPivot('category_id');
    }

    public function media()
    {
        return $this->belongsTo('App\Models\Media\Media');
    }

    /**
     * Return the user.
     */
    public function user()
    {
        return $this->belongsTo('App\Models\Access\User\User');
    }

    public function tags()
    {
        return $this->belongsToMany(Tag::class);
    }

    /**
     * Return the comments.
     */
    public function comments()
    {
        return $this->belongsToMany(Comment::class);
    }

    public function attachTags($tags)
    {
        if (is_array($tags)) {
            $tags = Tag::add($tags);
            $this->tags()->attach($tags);
        } else {
            $json_tags = json_decode($tags, true);
            if(is_array($json_tags)) {
                $tags = Arr::flatten($json_tags);
            }else{
                $tags = explode(',', $tags);
            }
            $this->attachTags($tags);
        }
    }

    public function syncTags($tags)
    {
        
        if (is_array($tags)) {
            $tags = Tag::add($tags);
            $this->tags()->sync($tags);
        } else {
            $json_tags = json_decode($tags, true);
            if(is_array($json_tags)) {
                $tags = Arr::flatten($json_tags);
            }else{
                $tags = explode(',', $tags);
            }
            $this->syncTags($tags);
        }
    }

    public function toFeedItem() : FeedItem
    {
        return FeedItem::create([
            'id' => $this->id,
            'title' => $this->title,
            'summary' => $this->content(30),
            'updated' => $this->created_at,
            'link' => route('post', $this->slug),
            'author' => $this->user->name,
        ]);
    }

    public static function getFeedItems()
    {
        return Post::where('type', 'post')->where('status', '1')->get();
    }

    public function toSitemapTag(): Url | string | array
    {
        return route('post', $this->slug);
    }
}
