<?php
namespace App\Http\Controllers\Frontend;

use DB;
use Setting;
use Illuminate\Http\Request;
use App\Models\Access\User\User;
use App\Models\Access\User\Profile;
use Illuminate\Support\Facades\Log;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Artisan;
use App\Repositories\Frontend\InstallRepository;
use App\Http\Requests\Frontend\InstallSettingsRequest;
use App\Http\Requests\Frontend\InstallSettingsEmailRequest;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Schema;

/**
 * Class InstallController
 * @package App\Http\Controllers\Frontend
 * @author Rawbinn Shrestha <rawbinnn@gmail.com>
 */

class InstallController extends Controller
{
    /**
     * @var InstallRepository
     */
    private $installRepository;

    /**
     * InstallController constructor.
     * @param InstallRepository $installRepository
     */
    public function __construct(InstallRepository $installRepository)
    {
        $this->installRepository = $installRepository;
    }

    /**
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function index()
    {
        $steps = ['welcome' => 'active'];
        return view('install.start', compact('steps'));
    }

    /**
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function requirements()
    {
        $requirements = $this->installRepository->getRequirements();
        $allLoaded = $this->installRepository->allRequirementsLoaded();

        $steps = [
            'welcome' => 'success_step',
            'requirements' => 'active',
        ];
        return view('install.requirements', compact('requirements', 'allLoaded', 'steps'));
    }

    /**
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\Http\RedirectResponse|\Illuminate\Routing\Redirector|\Illuminate\View\View
     */
    public function permissions()
    {
        if (!$this->installRepository->allRequirementsLoaded()) {
            return redirect('install/requirements');
        }

        $folders = $this->installRepository->getPermissions();
        $allGranted = $this->installRepository->allPermissionsGranted();

        $steps = [
            'welcome' => 'success_step',
            'requirements' => 'success_step',
            'permissions' => 'active',
        ];
        return view('install.permissions', compact('folders', 'allGranted', 'steps'));
    }

    /**
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\Http\RedirectResponse|\Illuminate\Routing\Redirector|\Illuminate\View\View
     */
    public function database()
    {
        if (!$this->installRepository->allRequirementsLoaded()) {
            return redirect('install/requirements');
        }

        if (!$this->installRepository->allPermissionsGranted()) {
            return redirect('install/permissions');
        }

        $steps = [
            'welcome' => 'success_step',
            'requirements' => 'success_step',
            'permissions' => 'success_step',
            'database' => 'active',
        ];

        return view('install.database', compact('steps'));
    }

    /**
     * @param Request $request
     * @return $this|\Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function installation(Request $request)
    {
        $request->validate([
            'host' => 'required',
            'username' => 'required',
            'database' => 'required'
        ]);
        if (!$this->installRepository->allRequirementsLoaded()) {
            return redirect('install/requirements');
        }

        if (!$this->installRepository->allPermissionsGranted()) {
            return redirect('install/permissions');
        }

        $dbCredentials = $request->only('host', 'username', 'password', 'database');
        
        if (!$this->installRepository->dbCredentialsAreValid($dbCredentials)) {
            return redirect('install/database')
                ->withInput(Arr::except($dbCredentials, 'password'))
                ->withErrors(trans("install.connection_established"));
        }
        $steps = [
            'welcome' => 'success_step',
            'requirements' => 'success_step',
            'permissions' => 'success_step',
            'database' => 'success_step',
            'installation' => 'active',
        ];
        return view('install.installation', compact('steps'));
    }

    /**
     * @return \Illuminate\Http\RedirectResponse|\Illuminate\Routing\Redirector
     */
    public function install()
    {
        try {
            Artisan::call('cache:clear');
            if(strlen(config('app.key')) != 51) {
                Artisan::call('key:generate');
            }
            Artisan::call('migrate', ['--force' => true]);
            Artisan::call('db:seed', ['--force' => true]);
            if(File::exists(public_path('storage'))) {
                File::deleteDirectory(public_path('storage'));
            }
            Artisan::call('storage:link');
            return redirect('install/settings');

        } catch (\Exception $e) {
            // @unlink(base_path('.env'));
            Artisan::call('migrate:reset', ['--force' => true]);
            Log::error($e->getMessage());
            Log::error($e->getTraceAsString());

            return redirect('install/error');
        }
    }

    /**
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function disable()
    {
        $foldersDisable = $this->installRepository->getDisablePermissions();
        $allDisableGranted = $this->installRepository->allDisablePermissionsGranted();
        $steps = [
            'welcome' => 'success_step',
            'requirements' => 'success_step',
            'permissions' => 'success_step',
            'database' => 'success_step',
            'installation' => 'success_step',
            'settings' => 'active',
        ];
        return view('install.disable', compact('foldersDisable', 'allDisableGranted', 'steps'));
    }

    /**
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function settings()
    {
        if(!Schema::hasTable('users')) { 
            Artisan::call('migrate:reset', ['--force' => true]);          
            return redirect('install');
        }
        if(DB::table('users')->count()){
            return redirect('complete');
        }
        $steps = [
            'welcome' => 'success_step',
            'requirements' => 'success_step',
            'permissions' => 'success_step',
            'database' => 'success_step',
            'installation' => 'success_step',
            'settings' => 'active',
        ];
        Setting::forgetAll();
        return view('install.settings', compact('steps'));
    }

    /**
     * @param InstallSettingsRequest $request
     * @return \Illuminate\Http\RedirectResponse|\Illuminate\Routing\Redirector
     */
    public function settingsSave(InstallSettingsRequest $request)
    {
        Setting::set('site-title', $request->site_title);
        Setting::set('site-tagline', $request->site_tagline);
        Setting::set('site-phone', $request->site_phone);
        Setting::set('site-address', $request->site_address);
        Setting::set('site-email', $request->site_email);
        Setting::set('theme-active', 'gear');

        $user = new User;
        $user->email = $request->email;
        $user->confirmation_code = md5(uniqid(mt_rand(), true));
        $user->status = 1;
        $user->password = bcrypt($request->password);
        $user->confirmed = 1;

        DB::transaction(function () use ($user, $request) {
            if ($user->save()) {
                /**
                 * Add the admin role to the first user
                 */
                $user->attachRole(1);
                Profile::create([
                        'user_id' => $user->id,
                        'firstname' => $request->first_name,
                        'lastname' => $request->last_name
                ]);
            }
        });

        return redirect('install/complete');
    }

    /**
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function settingsEmail()
    {
        $steps = [
            'welcome' => 'success_step',
            'requirements' => 'success_step',
            'permissions' => 'success_step',
            'database' => 'success_step',
            'installation' => 'success_step',
            'settings' => 'success_step',
            'mail_settings' => 'active'];
        return view('install.mail_settings', compact('steps'));
    }

    /**
     * @param InstallSettingsEmailRequest $request
     * @return $this|\Illuminate\Http\RedirectResponse|\Illuminate\Routing\Redirector
     */
    public function settingsEmailSave(InstallSettingsEmailRequest $request)
    {
        try {
            if ($request->email_driver == 'smtp') {
                // $transport = Swift_SmtpTransport::newInstance($request->email_host, $request->email_port, 'ssl');
                // $transport->setUsername($request->email_username);
                // $transport->setPassword($request->email_password);
                // $mailer = \Swift_Mailer::newInstance($transport);
                // $mailer->getTransport()->start();
            }
            foreach ($request->except('_token') as $key => $value) {
                Setting::set($key, $value);
            }
            
            file_put_contents(storage_path('installed'), 'Welcome to Larapress');

            return redirect('install/complete');

        } catch (Swift_TransportException $e) {
            return redirect()->back()->withErrors($e->getMessage());
        } catch (\Exception $e) {
            return redirect()->back()->withErrors($e->getMessage());
        }
    }

    /**
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function complete()
    {
        $steps = [
            'welcome' => 'success_step',
            'requirements' => 'success_step',
            'permissions' => 'success_step',
            'database' => 'success_step',
            'installation' => 'success_step',
            'settings' => 'success_step',
            'mail_settings' => 'success_step',
            'complete' => 'active'];
        $path = base_path('.env');
        $env = file($path);
        $env = str_replace('INSTALLED=false','INSTALLED=true', $env);
        $env = str_replace('APP_URL=','APP_URL='.basename($_SERVER['HTTP_HOST']), $env);
        file_put_contents($path, $env);

        return view('install.complete', compact('steps'));
    }

    /**
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function error()
    {
        return view('install.error');
    }
}
