<?php

namespace App\Http\Controllers\Frontend;

use App\Mail\ContactMail;
use App\Models\Comment\Comment;
use App\Models\Contact\Contact;
use App\Models\Gallery\Gallery;
use App\Modules\Service\Models\Service;
use App\Repositories\Frontend\BlogRepository;
use App\Repositories\Frontend\PageRepository;
use Artesaos\SEOTools\Facades\SEOTools;
use Illuminate\Http\Request;
use Mail;
use Rawbinn\Larapress\Facades\Larapress;
use Theme;
use Validator;

/**
 * Class HomeController.
 *
 * @author Rawbinn Shrestha <rawbinnn@gmail.com>
 */
class HomeController extends FrontendController
{
    /**
     * @var BlogRepository
     */
     protected $blogs;
    
     /**
     * @var PageRepository
     */
    protected $page;

    /**
     * HomeController constructor.
     */
    public function __construct(
        BlogRepository $blogs,
        PageRepository $page
        ) {
        $this->blogs = $blogs;
        $this->page = $page;
        parent::__construct();
    }

    /**
     * @return mixed
     */
    public function index(Request $request)
    {
        if(!config('access.INSTALLED', false)) {
            return redirect()->route('installer');
        }
        if ($request->has('search')) {
            $results = $this->blogs->search($request->search);

            return Theme::view('search', compact('results'));
        }

        SEOTools::setTitle(setting('site-title'));
        SEOTools::setDescription(setting('meta-description'));
        SEOTools::opengraph()->setUrl(url()->current());
        SEOTools::setCanonical(url()->current());
        SEOTools::opengraph()->addProperty('type', 'articles');
        SEOTools::jsonLd()->addImage(setting('site-logo'));

        // For contact form submission
        if ($request->has('larapressContactToken') && $request->has('formName')) {
            if($request->larapressContactToken != csrf_token()) {
                return redirect()->route('index')->withFlashDanger('Invalid token');
            }
            $this->page->sendMail($request->except(['larapressContactToken']));
            return redirect()->back();
        }
        // End contact form submission
        
        return Theme::view('index');
    }

    public function services(Request $request)
    {
        $service = Service::where('type', '=', 'service')->first();

        if ($request->has('search')) {
            $results = $this->blogs->search($request->search);

            return Theme::view('search', compact('results'));
        }

        return Theme::view('services', compact('service'));
    }

    /**
     * @return \Illuminate\Http\JsonResponse
     */
    public function submitComment(Request $request)
    {
        if (!$request->ajax()) {
            return response()->json('Accepts only ajax request.');
        }
        Validator::make($request->all(), [
            'comment' => 'required',
        ])->validate();
        $blog = $this->blogs->findById($request->post_id);
        if ($blog == null) {
            return response()->json('Blog not found.');
        }
        $comment = new Comment();
        $comment->comment = filter_var($request->comment, FILTER_SANITIZE_STRING);
        if (auth()->guest()) {
            $comment->guest = true;
            $comment->user_id = null;
            $name = 'Anomymous User';
        } else {
            $comment->user_id = auth()->user()->id;
            $comment->guest = false;
            $name = auth()->user()->name;
        }
        $comment->parent = ($request->has('parent')) ? $request->parent : null;
        if ($comment->save()) {
            $blog->comments()->save($comment);
        }

        return response()->json(['status' => 'success', 'comment' => ['name' => $name, 'content' => $comment->comment]]);
    }

    public function submitContact(Request $request)
    {
        $data = $request->except('_token');
        $contact = Contact::where('slug', $data['contactFormName'])->first();
        if ($contact != null) {
            $contact->email = $contact->receiver_email;
            $contact->name = 'Saafalta';
            unset($data['contactFormName']);
            try {
                Mail::to($contact)->send(new ContactMail($data));
            } catch (\Exception $e) {
                return response()->json(['status' => false, 'message' => $e->getMessage()]);
            }

            return response()->json(['status' => true, 'message' => $contact->success_message]);
        }
    }

    public function gallery($slug)
	{
		$gallery = Gallery::where('slug', $slug)->firstOrFail();
		if($gallery->type == 'video') {
			
			return Theme::view('videos', compact('gallery'));
		}
		return Theme::view('images', compact('gallery'));
	}

    public function changeLanguage(Request $request)
    {
        $locales = locales();
        if(in_array($request->locale, $locales)) {
            $request->session()->put('locale', $request->locale);
        }
    }

}
