<?php

namespace App\Http\Controllers\Backend;

use App\Models\Menu\MenuItem;
use App\Repositories\Backend\BlogRepository;
use App\Repositories\Backend\MenuRepository;
use App\Repositories\Backend\PageRepository;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Menu;
use Setting;
use Theme;
use View;
use Eventy;

/**
 * Class MenuController.
 *
 * @author Rawbinn Shrestha <rawbinnn@gmail.com>
 */
class MenuController extends BaseController
{
    /**
     * @var MenuRepository
     */
    protected $menu;

    /**
     * @var PageRepository
     */
    protected $page;

    /**
     * @var PostRepository
     */
    protected $post;

    /**
     * @param PageRepository $menu
     */
    public function __construct(MenuRepository $menu, PageRepository $page, BlogRepository $post)
    {
        $this->menu = $menu;
        $this->page = $page;
        $this->post = $post;
    }

    /**
     * @param null
     *
     * @return mixed
     */
    public function index()
    {
        $data['groups'] = $this->menu->getMenuGroup(); // Returns menu group for select option
        $data['menu_groups'] = $this->menu->allMenu(); // Returns all menu with child
        $data['pages'] = $this->page->getAll('DESC');
        $data['posts'] = $this->post->getAll('DESC');
        $data['menu_locations'] = $this->getMenuLocations();
        $data['theme_menu'] = setting('theme-active', config('themes.active')).'-menu';
        return view($this->theme('menus.index'), $data);
    }

    /**
     * @return mixed
     */
    public function store(Request $request)
    {
        MenuModel::create($request->all());
    }

    public function getMenuLocations()
    {
        // $menuLocations = Theme::getFunctionProperty(setting('theme-active', config('themes.active')), 'menus');
        $menuLocations = Eventy::filter('regsiter_nav_menus', ['primary-menu' => 'Primary Menu']);
        return $menuLocations;
    }

    /**
     * @param $id
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function edit($id)
    {
        $menu_item = $this->menu->find($id);
        $menus = $this->menu->allMenu(); // Returns top level menu items(Root menu items)

        return response()->json(view($this->theme('menus.edit'), compact('menu_item', 'menus'))->render());
    }

    /**
     * @return \Illuminate\Http\JsonResponse
     */
    public function update(Request $request)
    {
        $this->validate($request, [
            'menu_id' => 'required|integer',
            'parent_id' => 'required|integer',
        ]);

        $data = [];
        $data['toplevel'] = false;
        $menuItem = MenuItem::find($request->menu_id);
        $menu = $this->menu->update($menuItem, $request->except('menu_id', '_token'));

        return response()->json($data);
    }

    /**
     * @return mixed
     *
     * @throws \App\Exceptions\GeneralException
     */
    public function storeGroup(Request $request)
    {
        $this->menu->createGroup($request->all());

        return redirect()->route('admin.menu.index')->withFlashSuccess('Menu Group successfully added.');
    }

    /**
     * Called from ajax.
     */
    public function add(Request $request)
    {
        $data = [];
        $data['toplevel'] = false;
        if ($request->has('page_id') && is_array($request->page_id) && count($request->page_id) > 0) {
            $pages = $this->page->getPagesDataForMenu($request->page_id);
            foreach ($pages as $page) {
                $menu = $this->menu->create(['title' => $page->title, 'url' => route('post', $page->slug), 'icon' => '', 'parent_id' => 0]);
            }
        } else {
            $menu = $this->menu->create($request->all());
        }

        $selected_menu = $this->menu->selectMenu($menu->menu_id);
        if ($selected_menu) {
            $menugroups = $this->menu->getMenuItems($menu->menu_id);
            if (!$menugroups->isEmpty()) {
                $data['menuView'] = View::make($this->theme('menus.hierarchy'), ['menugroups' => $menugroups])->render();
                $allMenuGroups = $this->menu->allMenu(); // Returns all menu with child
                $data['menuSelect'] = View::make($this->theme('menus.selectMenuList'), ['menus' => $allMenuGroups])->render();

                return response()->json($data);
            }

            return response()->json('No menu yet...');
        }

        return response()->json($data);
    }

    public function settings(Request $request)
    {
        $validator = Validator::make($request->all(), [
			'menu' => 'required|integer',
        ]);
        if ($validator->fails()) {
            $errors = $validator->messages()->all();
            return response()->json(['status' => false, 'message' => $errors[0], 'data' => []]);
		}
        $menuSlug = [];
        $theme_menu = setting('theme-active', config('themes.active')).'-menu';
        $settings = settings($theme_menu, []);
        $menu_locations = $this->getMenuLocations();
        foreach($menu_locations as $menu_slug => $menu_value) {
            $setting = $settings[$menu_slug] ?? [];
            if($request->has('display_location')) {
                $display_location = $request->display_location;
                
                if(isset($display_location[$menu_slug]) && $display_location[$menu_slug] == 'on') {
                    $settings[$menu_slug] = $request->menu;
                } else {
                    if(isset($settings[$menu_slug]) && ($settings[$menu_slug] == $request->menu)) {
                        $settings[$menu_slug] = '';
                    }
                }
                
            } else {
                $settings[$menu_slug] = '';
            }
        }

        Setting::set($theme_menu, json_encode($settings));
        return response()->json(['status' =>true]);
    }

    /**
     * @return \Illuminate\Http\JsonResponse
     */
    public function select(Request $request)
    {
        $this->validate($request, [
            'menu_id' => 'integer',
        ]);
        $menu = $this->menu->selectMenu($request->menu_id);
        $menu_id = $request->menu_id;

        if ($menu) {
            $menugroups = $this->menu->getMenuItems($request->menu_id);
            if (!$menugroups->isEmpty()) {
                $menu_locations = $this->getMenuLocations();
                $theme_menu = setting('theme-active', config('themes.active')).'-menu';
                return response()->json([
                    'menu' => view($this->theme('menus.hierarchy'), ['menugroups' => $menugroups, 'menu_id' => $menu_id])->render(),
                    'settings' => view($this->theme('menus.menuSettings'), [
                        'menu_id' => $menu_id, 
                        'menu_locations' => $menu_locations,
                        'theme_menu' => $theme_menu
                        ])->render()
                ]);
            }

            return response()->json([
                'menu' => 'No menu yet...',
                'settings' => ''
            ]);
        }

        return response()->json(false);
    }

    /**
     * @return \Illuminate\Http\JsonResponse
     */
    public function hierarchy(Request $request)
    {
        $menu = $this->menu->updateHierarchy($request->data);
        if ($menu) {
            return response()->json(['status' => true]);
        }
    }

    /**
     * @return \Illuminate\Http\JsonResponse
     */
    public function delete(MenuItem $menu, Request $request)
    {
        $msg = $this->menu->delete($menu);
        if ($msg === true) {
            return response()->json(['status' => true, 'menu_id' => $menu->id]);
        }

        return response()->json(['status' => false, 'msg' => $msg]);
    }
}
