<?php
namespace App\Http\Controllers\Backend;

use Validator;
use Illuminate\Http\Request;
use App\Models\Gallery\Gallery;
use Illuminate\Routing\Controller;
use App\Models\Gallery\GalleryItem;
use DataTables;

/**
 * Class GalleryItemController
 * @package App\Http\Controllers\Backend
 * @author Rawbinn Shrestha <rawbinnn@gmail.com>
 */
class GalleryItemController extends BaseController
{
    /**
     * @param $id
     * @return View
     */
    public function index($id)
    {
        $gallery = Gallery::find($id);
        return view($this->theme('gallery.items.index'))->withGallery($gallery);
    }

    /**
     * @param $id
     * @return View
     */
    public function create($id)
    {
        $gallery = Gallery::find($id);
        return view($this->theme('gallery.items.create'))->withGallery($gallery);
    }

    /**
     * @param $id
     * @param Request $request
     * @return mixed
     */
    public function store($id, Request $request)
    {
        $request->validate([
            'media_id' => 'required',
        ]);
        $gallery = Gallery::findOrFail($id);

        try {
            $media_id_array = explode(',', $request->media_id);
            foreach ($media_id_array as $media_id) {
                $galleryItem = new GalleryItem();
                $galleryItem->title = $request->title;
                $galleryItem->description = $request->description;
                $galleryItem->image_id = $media_id;
                $galleryItem->gallery_id = $gallery->id;
                $galleryItem->save();
            }
        } catch (\Exception $e) {
            return redirect()->route('admin.gallery.item.index', $gallery->id)->withFlashDanger($e->getMessage());
        }

        return redirect()->route('admin.gallery.item.index', $gallery->id)->withFlashSuccess('Gallery item added successfully.');
    }

    /**
     * @param $id
     * @return View
     */
    public function edit($id)
    {
        $gallery = GalleryItem::find($id);
        return view($this->theme('gallery.items.edit'))->withGallery($gallery);
    }

    /**
     * @param $id
     * @param Request $request
     * @return mixed
     */
    public function update($id, Request $request)
    {
        Validator::make($request->all(), [
            'title' => 'required',
        ])->validate();

        $gallery = GalleryItem::find($id);
        $gallery->title = $request->title;
        $gallery->description = $request->description;
        if ($request->media_id != '0') {
            $gallery->image_id = $request->media_id;
        }

        if ($gallery->save()) {
            return redirect()->route('admin.gallery.item.index', $gallery->gallery_id)->withFlashSuccess('Gallery item updated successfully.');
        }
    }

    /**
     * @param $id
     * @return mixed
     */
    public function delete($id)
    {
        $gallery = GalleryItem::find($id);
        if ($gallery->delete()) {
            return redirect()->back()->withFlashDanger('Gallery item deleted successfully.');
        }
        return redirect()->back()->withFlashDanger('There is some error occured during deleting data.');
    }

    /**
     * @return DataTables
     */
    public function getForDatatables($id)
    {
        $gallery = Gallery::find($id);
        $galleryItems = $gallery->items;
        return DataTables::of($galleryItems)
            ->addColumn('image', function ($gallery) {
                return '<img src="' . image($gallery->image_id, false) . '" width="130px">';
            })
            ->editColumn('created_at', function ($gallery) {
                return $gallery->created_at;
            })
            ->editColumn('updated_at', function ($gallery) {
                return $gallery->updated_at;
            })
            ->addColumn('actions', function ($gallery) {
                return $gallery->action_buttons;
            })
            ->escapeColumns([])
            ->make(true);
    }
}
