<?php

namespace App\Http\Controllers\Backend;

use App\Models\Category\Category;
use App\Repositories\Backend\CategoryRepository;
use Illuminate\Http\Request;
use View;

/**
 * Class CategoryController.
 *
 * @author Rawbinn Shrestha <rawbinnn@gmail.com>
 */
class CategoryController extends BaseController
{
    /**
     * @var CategoryRepository
     */
    protected $category;

    public function __construct(CategoryRepository $category)
    {
        $this->category = $category;
    }

    /**
     * @return \Illuminate\View\View
     */
    public function index()
    {
    }

    /**
     * Returns form view to create category and also it displays all category.
     *
     * @param null
     *
     * @return mixed
     */
    public function create()
    {
        $categories = $this->category->getTree(); // Returns tree of approved categories
        $parents = $this->category->getCategoriesWithChild(); // Returns parent categories with limit 10

        return view($this->theme('category.create'), compact('categories', 'parents'));
    }

    /**
     * Store category and redirect back to same page.
     *
     * @return mixed
     */
    public function store(Request $request)
    {
        $request->validate([
            'title' => 'required',
        ]);
        $this->category->create($request->all());

        return redirect()->back()->withFlashSuccess(trans('alerts.backend.general.created', ['attribute' => 'category']));
    }

    /**
     * @param $id
     *
     * @return \Illuminate\Http\JsonResponse
     *
     * @throws \Throwable
     */
    public function edit($id, Request $request)
    {
        $lang = lang($request->lang);
        $categories = $this->category->getTree('product'); // Returns tree of approved categories
        $category = $this->category->find($id);

        return view('admin.category.edit', compact('categories', 'category', 'lang'));
    }

    /**
     * @return \Illuminate\Http\JsonResponse
     *
     * @throws \App\Exceptions\GeneralException
     */
    public function update($id, Request $request)
    {
        $lang = lang($request->lang);
        $request->validate([
            $lang.'.title' => 'required',
            'slug' => 'required'
        ]);
        $category = $this->category->find($id);
        $this->category->update($category, $request->all());

        return redirect()->back()->withFlashSuccess(trans('alerts.backend.general.updated', ['attribute' => 'category']));
    }

    /**
     * @param int $id
     *
     * @return json
     */
    public function delete(Category $category)
    {
        $this->category->delete($category);

        return redirect()->route('admin.category.create')->withFlashSuccess(trans('alerts.backend.general.deleted_permanently', ['attribute' => 'category']));
    
    }

    public function deleteSelected(Request $request)
    {
        $this->category->deleteSelected($request->id);
        return response()->json(['status' => true, 'message' => 'Selected categories deleted successfully.']);
    }
}
